/** @file   barrellobject.h
 * @brief   Declaration of BarrellObject - class.
 * @version $Revision: 1.1.1.1 $
 * @date    $Date: 2006/01/21 23:02:41 $
 * @author  Tomi Lamminsaari
 */

#ifndef H_WWW_BARRELLOBJECT_H
#define H_WWW_BARRELLOBJECT_H

#include "gameobject.h"
#include <list>

namespace WeWantWar {

/** @class  BarrellObject
 * @brief   Represents the rolling and exploding barrell.
 * @author  Tomi Lamminsaari
 *
 * When the barrells are moving, they explode as the collide to the tilemap.
 * And this explosion can modify the tilemap. The information about how to
 * modify is read from a certain file. The modification will be targeted
 * to map layer 0.
 */
class BarrellObject : public GameObject
{
public:

  ///
  /// Constants, datatype and static members
  /// ======================================


  ///
  /// Constructors, destructor and operators
  /// ======================================

	/** Constructor.
   */
	BarrellObject();


	/** Destructor
   */
	virtual ~BarrellObject();

private:

	/** Copy constructor.
   * @param     rO                Reference to another BarrellObject
   */
  BarrellObject( const BarrellObject& rO );

	/** Assignment operator
   * @param     rO                Reference to another BarrellObject
   * @return    Reference to us.
   */
  BarrellObject& operator = ( const BarrellObject& rO );

public:



  ///
  /// Methods
  /// =======

  /** Updates this object.
   */
  virtual void update();
  
  /** Redraws this
   * @param     aQueue            Pointer to redraw queue
   */
  virtual void redraw( RedrawQueue* aQueue );
  
  /** Kills this object
   */
  virtual void kill();
  
  /** Makes sound
   * @param     aSoundId          Id code of the sound we should make.
   */
  virtual void makeSound( GameObject::SoundID aSoundId ) const;
  
  /** Handles the bullet hits
   * @param     aBullet           Pointer to bullet that hit us.
   * @return    <code>true</code> if we accept the bullet hit
   */
  virtual bool hitByBullet( Bullet* aBullet );
  
  /** Messageport so that we can trigger the rolling phase.
   * @param     aMessage          Reference to the message
   */
  virtual void messagePort( const ObjectMessage& aMessage );
  
  /** Reads the modification data.
   * @param     aFilename         Name of the file where to read. If the given
   *                              filename equals "NONE", we do not load
   *                              anything.
   * @return    Nonzero if fails.
   */
  int readModifyData( const std::string& aFilename );
  

  ///
  /// Getter methods
  /// ==============
  
  /** Returns the type of this object.
   * @return    ObjectID::TYPE_BARRELL
   */
  virtual ObjectID::Type objectType() const;
  
  /** Are we in reloading state
   * @return    Always false
   */
  virtual bool reloading() const;
  


protected:

  /** Sets the requested animation.
   * @param     aAnimId           Id of the animation to set up
   */
  void setCorrectAnimation( int aAnimId );
  
  /** Tells if we're about to collide to the tilemap during the next update
   * round.
   * @return    <code>true</code> if we will collide next time.
   */
  bool collisionNextTime();
  
  /** Modifies the tilemap.
   */
  void applyMapModification();
  
  
  
  /** The tiles being modified during the explosion are stored in a
   * structure of this type.
   */
  struct ModifyItem {
    /** X coordinate of the tile being modified in tiles from topleft corner. */
    int iPosX;
    /** Y coordinate of the tile being modified in tiles from topleft corner. */
    int iPosY;
    /** The new tilenumber we will place on that position. */
    int iNewTile;
  };
  
  
  ///
  /// Members
  /// =======

  /** The movement velocity */
  eng2d::Vec2D  iVelocity;
  
  /** Number of fragments we should emit when exploding */
  int   iFragmentCount;
  
  /** How much damage does each fragment do. */
  int   iFragmentDamage;

  /** List of the tiles we will modify as we explode. */
  std::list< ModifyItem* >  iModifyTiles;

private:

  ///
  /// Private members
  /// ===============

};

};  // end of namespace

#endif


/**
 * Version history
 * ===============
 * $Log: barrellobject.h,v $
 * Revision 1.1.1.1  2006/01/21 23:02:41  lamminsa
 * no message
 *
 * Revision 1.0  2005-11-06 01:16:48+02  lamminsa
 * Initial revision
 *
 */
 
